# -*- coding: utf-8 -*-
"""
AutoPrint80mm (Windows Service) — tudo em uma pasta
- Consulta MySQL a cada POLL_SECONDS
- Se registro ainda não impresso, manda RAW para impressora 80mm (sem diálogo)
- Marca como impresso no banco
- Roda como serviço Windows (pywin32)
- Logs e .env ficam na mesma pasta do script
- Modo simulação opcional (salva arquivo ao invés de imprimir)

Requisitos:
  pip install -r requirements.txt
Comandos úteis:
  python autoprint_service.py debug        -> roda 1 ciclo no console
  python autoprint_service.py printers     -> lista impressoras disponíveis
  python autoprint_service.py install      -> instala o serviço do Windows
  python autoprint_service.py start        -> inicia o serviço
  python autoprint_service.py stop         -> para o serviço
  python autoprint_service.py remove       -> remove o serviço
"""

import os
import sys
import time
import socket
import traceback
from pathlib import Path
from datetime import datetime

from dotenv import load_dotenv
import mysql.connector as mysql
from mysql.connector import Error as MySQLError

# pywin32 (serviço e impressão RAW)
import win32serviceutil
import win32service
import win32event
import servicemanager
import win32print

# ===================== Paths & Const =====================
APP_DIR = Path(__file__).resolve().parent
LOG_DIR = APP_DIR / "logs"
OUT_DIR = APP_DIR / "out"
LOG_DIR.mkdir(parents=True, exist_ok=True)
OUT_DIR.mkdir(parents=True, exist_ok=True)

LOG_FILE = LOG_DIR / "service.log"
ENV_FILE = APP_DIR / ".env"

APP_NAME = "AutoPrint80mm"
APP_DISPLAY_NAME = "AutoPrint 80mm Service"
APP_DESCRIPTION = "Consulta MySQL e imprime automaticamente em impressora térmica 80mm."

# ===================== Utils =====================
def log(msg: str):
    ts = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    with open(LOG_FILE, "a", encoding="utf-8") as f:
        f.write(f"[{ts}] {msg}\n")

def log_exc(prefix="ERRO"):
    log(f"{prefix}: {traceback.format_exc()}")

def load_config():
    # Carrega .env da mesma pasta
    if ENV_FILE.exists():
        load_dotenv(ENV_FILE)
    else:
        load_dotenv()

    cfg = {
        # MySQL
        "DB_HOST": os.getenv("DB_HOST", "127.0.0.1"),
        "DB_PORT": int(os.getenv("DB_PORT", "3306")),
        "DB_USER": os.getenv("DB_USER", "root"),
        "DB_PASS": os.getenv("DB_PASS", ""),
        "DB_NAME": os.getenv("DB_NAME", "test"),
        # Tabela/colunas
        "TABLE": os.getenv("TABLE", "tickets"),
        "COL_ID": os.getenv("COL_ID", "id"),
        "COL_TEXT": os.getenv("COL_TEXT", "conteudo"),
        "COL_PRINTED_AT": os.getenv("COL_PRINTED_AT", "impresso_em"),
        # Poll
        "POLL_SECONDS": int(os.getenv("POLL_SECONDS", "5")),
        "BATCH_LIMIT": int(os.getenv("BATCH_LIMIT", "10")),
        # Impressão
        "PRINTER_NAME": os.getenv("PRINTER_NAME", ""),   # vazio = padrão do Windows
        "TEXT_ENCODING": os.getenv("TEXT_ENCODING", "cp858"),
        "MAX_COLS": int(os.getenv("MAX_COLS", "42")),
        "ADD_HEADER": os.getenv("ADD_HEADER", "1") in ("1","true","True"),
        "HEADER_TEXT": os.getenv("HEADER_TEXT", "Seu Negócio"),
        "ADD_ESC_POS_CUT": os.getenv("ADD_ESC_POS_CUT", "1") in ("1","true","True"),
        # Simulação (não imprime; salva payload em /out)
        "SIMULATE_PRINT": os.getenv("SIMULATE_PRINT", "0") in ("1","true","True"),
    }
    return cfg

# ===================== DB =====================
def connect_db(cfg):
    return mysql.connect(
        host=cfg["DB_HOST"],
        port=cfg["DB_PORT"],
        user=cfg["DB_USER"],
        password=cfg["DB_PASS"],
        database=cfg["DB_NAME"],
        autocommit=False,
        connection_timeout=10,
    )

def fetch_jobs(conn, cfg):
    sql = (
        f"SELECT {cfg['COL_ID']}, {cfg['COL_TEXT']} "
        f"FROM {cfg['TABLE']} "
        f"WHERE {cfg['COL_PRINTED_AT']} IS NULL "
        f"ORDER BY {cfg['COL_ID']} ASC "
        f"LIMIT %s"
    )
    with conn.cursor() as cur:
        cur.execute(sql, (cfg["BATCH_LIMIT"],))
        rows = cur.fetchall()
    return [{"id": r[0], "text": r[1] or ""} for r in rows]

def mark_printed(conn, cfg, ids):
    if not ids:
        return
    placeholders = ", ".join(["%s"] * len(ids))
    sql = (
        f"UPDATE {cfg['TABLE']} "
        f"SET {cfg['COL_PRINTED_AT']} = NOW() "
        f"WHERE {cfg['COL_ID']} IN ({placeholders})"
    )
    with conn.cursor() as cur:
        cur.execute(sql, tuple(ids))
    conn.commit()

# ===================== Impressão =====================
def escpos_cut(full=True):
    # GS V 0  (pode variar por modelo)
    return b"\x1d\x56\x00" if full else b"\x1d\x56\x01"

def wrap_cols(text: str, max_cols: int):
    lines = []
    for raw in (text or "").splitlines():
        line = raw.replace("\t", " ")
        while len(line) > max_cols:
            lines.append(line[:max_cols])
            line = line[max_cols:]
        lines.append(line)
    return "\n".join(lines)

def build_payload(text: str, cfg):
    parts = []
    if cfg["ADD_HEADER"]:
        header = cfg["HEADER_TEXT"].strip()
        if header:
            parts.append(header)
            parts.append("-" * min(cfg["MAX_COLS"], max(10, len(header))))
    parts.append(wrap_cols(text, cfg["MAX_COLS"]))
    parts.append("\n\n")  # alimenta o papel
    data = "\n".join(parts)
    payload = data.encode(cfg["TEXT_ENCODING"], errors="replace")
    if cfg["ADD_ESC_POS_CUT"]:
        payload += b"\n\n" + escpos_cut(True)
    else:
        payload += b"\n\n"
    return payload

def print_raw(payload: bytes, cfg):
    if cfg["SIMULATE_PRINT"]:
        # salva payload para inspecionar
        ts = datetime.now().strftime("%Y%m%d_%H%M%S_%f")
        out = OUT_DIR / f"print_{ts}.bin"
        with open(out, "wb") as f:
            f.write(payload)
        return True, f"simulado -> {out}"
    # Real printing via RAW spool
    try:
        printer_name = cfg["PRINTER_NAME"] or win32print.GetDefaultPrinter()
        hPrinter = win32print.OpenPrinter(printer_name)
        try:
            job = win32print.StartDocPrinter(hPrinter, 1, ("AutoPrint80mm", None, "RAW"))
            try:
                win32print.StartPagePrinter(hPrinter)
                win32print.WritePrinter(hPrinter, payload)
                win32print.EndPagePrinter(hPrinter)
            finally:
                win32print.EndDocPrinter(hPrinter)
        finally:
            win32print.ClosePrinter(hPrinter)
        return True, ""
    except Exception as e:
        return False, str(e)

# ===================== Loop =====================
def run_cycle(cfg):
    conn = None
    try:
        if not conn or (conn and not conn.is_connected()):
            conn = connect_db(cfg)

        jobs = fetch_jobs(conn, cfg)
        if jobs:
            log(f"Encontrados {len(jobs)} registro(s) para imprimir.")
        printed_ids = []

        for j in jobs:
            payload = build_payload(j["text"], cfg)
            ok, err = print_raw(payload, cfg)
            if ok:
                printed_ids.append(j["id"])
                log(f"Impresso id={j['id']}")
            else:
                log(f"Falha impressão id={j['id']}: {err}")

        if printed_ids:
            try:
                mark_printed(conn, cfg, printed_ids)
                log(f"Atualizado status de {len(printed_ids)} registro(s).")
            except Exception:
                log_exc("Erro ao atualizar status")
                conn.rollback()
        if not jobs:
            log("Nada a imprimir.")
    finally:
        try:
            if conn:
                conn.close()
        except Exception:
            pass

def run_loop(stop_event):
    cfg = load_config()
    log("=== AutoPrint iniciado ===")
    try:
        printer = cfg["PRINTER_NAME"] or win32print.GetDefaultPrinter()
    except Exception:
        printer = "(indefinida)"
    log(f"Impressora: {printer} | Simulação: {cfg['SIMULATE_PRINT']} | Poll: {cfg['POLL_SECONDS']}s")

    while not stop_event.is_set():
        try:
            run_cycle(cfg)
        except MySQLError:
            log_exc("Erro MySQL")
            time.sleep(3)
        except Exception:
            log_exc("Erro no loop")
        time.sleep(cfg["POLL_SECONDS"])

    log("=== AutoPrint finalizado ===")

# ===================== Serviço Windows =====================
class AutoPrintService(win32serviceutil.ServiceFramework):
    _svc_name_ = APP_NAME
    _svc_display_name_ = APP_DISPLAY_NAME
    _svc_description_ = APP_DESCRIPTION

    def __init__(self, args):
        win32serviceutil.ServiceFramework.__init__(self, args)
        self.stop_event = win32event.CreateEvent(None, 0, 0, None)

    def SvcStop(self):
        self.ReportServiceStatus(win32service.SERVICE_STOP_PENDING)
        win32event.SetEvent(self.stop_event)

    def SvcDoRun(self):
        servicemanager.LogMsg(
            servicemanager.EVENTLOG_INFORMATION_TYPE,
            servicemanager.PYS_SERVICE_STARTED,
            (self._svc_name_, "")
        )
        run_loop(self.stop_event)

# ===================== CLI Aux =====================
def list_printers():
    flags = 2  # PRINTER_ENUM_LOCAL
    printers = win32print.EnumPrinters(flags)
    print("Impressoras instaladas:")
    for p in printers:
        # p[2] é o nome da impressora
        print(" -", p[2])

if __name__ == "__main__":
    if len(sys.argv) >= 2 and sys.argv[1].lower() == "debug":
        # roda uma vez no console
        try:
            run_cycle(load_config())
        except KeyboardInterrupt:
            pass
        sys.exit(0)

    if len(sys.argv) >= 2 and sys.argv[1].lower() == "printers":
        list_printers()
        sys.exit(0)

    # Comandos do serviço (install/start/stop/remove)
    win32serviceutil.HandleCommandLine(AutoPrintService)
